/**
* \file    CRingBuffer.h
* \brief   Ring buffer class (adapted from Ctrlsys code)
* \author  Larry Salant
* \date    05/28/2009
**/
#include <stdlib.h>

#ifndef __CRingBuffer_H__
#define __CRingBuffer_H__

#define RING_BUFFER_OVERFLOW      -500   // indicates that data is lost
#define RING_BUFFER_EMPTY      -1
#define RING_BUFFER_BRK_RCVED      -501
#define RING_BUFFER_INVALIDATE_PACKET      -502
#define RING_BUFFER_SEND_BRK      -503
#define RING_BUFFER_END_OF_PACKET      -504

class CRingBuffer
{
protected:
    unsigned char *m_ucBufPtr;    // buffer pointer
    int            m_iBufSize;    // buffer size
    int            m_iBuffInIdx;  // index to input character
    int            m_iBuffOutIdx; // index to output character

public:
    CRingBuffer(int buf_size)  // allocates buffer space for buf_size in bytes
    {
        m_iBuffInIdx = m_iBuffOutIdx = 0;
        m_ucBufPtr = new unsigned char[buf_size];
        if ( !m_ucBufPtr ) 
            m_iBufSize = 0;
        else
            m_iBufSize = buf_size;
    }

    ~CRingBuffer()
    {
        delete[] m_ucBufPtr;
    }

    // clears the buffer
    void Clear(void ) { m_iBuffInIdx = m_iBuffOutIdx = 0;};  // resets all pointers and frees all space

    // returns a pointer to the beginning of the buffer
    unsigned char  *FirstByte(void ) { return (unsigned char  *)&m_ucBufPtr[0];};

    // adds one character to the buffer, non- blocking mode
    void AddToBuffer( unsigned char byte ) 
    {
        m_ucBufPtr[m_iBuffInIdx++] = byte;
        m_iBuffInIdx %= m_iBufSize;
    };

    // adds len characters to the buffer
    void AddToBuffer( unsigned char *str_pointer, int len ) 
    { 
        while (len) 
        {
            AddToBuffer(*str_pointer++);
            len--;
        }
    };
    
    // removes one character from the buffer and returns it
    int GetFromBuffer(void ) 
    {
        int byte;
        if ( m_iBuffOutIdx == m_iBuffInIdx) 
            return RING_BUFFER_EMPTY;

        byte = m_ucBufPtr[m_iBuffOutIdx++];
        m_iBuffOutIdx %= m_iBufSize;
        return byte;
    }; 

    // removes Len character from the buffer and copies them into caller's buffer
    // returns number of characters copied
    int GetFromBuffer( unsigned char *str_pointer, int len ) 
    {
        int cnt = 0;
        int byte;

        while ( (cnt < len) && ((byte = GetFromBuffer()) >= 0) )
            str_pointer[cnt++] = (unsigned char)byte;
        return cnt;
    };

    // return amount of space available in buffer
    int GetFreeSpace(void) 
    {
        if ( m_iBuffInIdx < m_iBuffOutIdx ) 
            return (m_iBuffOutIdx - m_iBuffInIdx);
        else 
            return (m_iBuffOutIdx - m_iBuffInIdx + m_iBufSize );
    };

};

#endif
